/*********************************************************************

          ps.c: PolhemusSensor device driver

                                        Kenji with Tail

*********************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termio.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/time.h>
#include <stropts.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <errno.h>
#include <termios.h>

#include "ps.h"


#define _DEBUG_ 1


/*********          Serial Port comunication driver          ********/
/* Public constants for x86(?)... */
#define OFLAGS  (OPOST | OLCUC | ONLCR | OCRNL | ONOCR  \
                        | ONLRET | OFILL | OFDEL)
#define LFLAGS  (ICANON | ISIG | XCASE | ECHO | ECHOE | ECHOK  \
                        | ECHONL | NOFLSH | TOSTOP | ECHOCTL  \
                        | ECHOPRT | ECHOKE | IEXTEN)
#define IFLAGS  (IGNBRK | BRKINT | IGNPAR | PARMRK | INPCK  \
                        | INLCR | ICRNL | IGNCR | IUCLC | ISTRIP  \
                        | IXON | IXOFF | IMAXBEL)




#define MIN_TIMEOUT 1
#define TIM_TIMEOUT 0

#define CTL_COM(n) ((n) - 0x40)
#define CR_COM 13
#define LF_COM 10
#define NUL_COM 0


#define NUM_COM	(4+1)	/* number of port is 4, 0-4 */

static	int		port_ref = -1;
static	struct	termios	old_setup;


/* OpenSP_com() opens the given serial port with specified baud rate
 *    Always uses 8 data bits, 1 stop bit, no parity.
 *    Assumes port number is valid and baud rate has been fixed up.
 *    Returns False (zero) if can't open the port.
 */
static int OpenSP_com(char *port_name, long int baud)
{
	int		result = 0;	/* Assume the port will not be accessible */
	tcflag_t	baud_code;
	struct	termios	new_setup;

	if (port_ref != -1){
		fprintf(stderr, "serial port has been opened");
		return result;
	}

	port_ref = open(port_name, O_RDWR);

	/* If port and its parameters are accessible, reconfigure for Probe */
	if (port_ref >= 0){

		/* Find appropriate baud rate flags */
		switch(baud){
		    case 0L:
			baud_code = B0;		break; /* down ER signal, close */
		    case 50L:
			baud_code = B50;	break;
		    case 75L:
			baud_code = B75;	break;
		    case 110L:
			baud_code = B110;	break;
		    case 134L:
			baud_code = B134;	break;
		    case 150L:
			baud_code = B150;	break;
		    case 200L:
			baud_code = B200;	break;
		    case 300L:
			baud_code = B300;	break;
		    case 600L:
			baud_code = B600;	break;
		    case 1200L:
			baud_code = B1200;	break;
		    case 1800L:
			baud_code = B1800;	break;
		    case 2400L:
			baud_code = B2400;	break;
		    case 4800L:
			baud_code = B4800;	break;
		    case 9600L:
			baud_code = B9600;	break;
		    case 19200L:
			baud_code = B19200;	break;
		    case 38400L:
			baud_code = B38400;	break;
		    case 'A':
			baud_code = EXTA;	break;
		    case 'B':
			baud_code = EXTB;	break;
		    default:
			fprintf(stderr, "OpenSP_com: bad baud rate: %ld\n", baud);
		}

		tcgetattr( port_ref, &old_setup);
		/* Save original set-up */
		new_setup = old_setup;

		/* Turn off all post-processing and char translation */
		new_setup.c_oflag = ~OFLAGS;
		new_setup.c_lflag = ~LFLAGS;
		new_setup.c_iflag = ~IFLAGS;

		/* Set for no waiting during char read */
		new_setup.c_cc[VMIN] = MIN_TIMEOUT;
		new_setup.c_cc[VTIME] = TIM_TIMEOUT;

		/* Set for baud rate, 8 data bits, 1 stop bit, no parity */
		new_setup.c_cflag |= CS8 | CREAD;
		cfsetospeed( &new_setup, baud_code );
		cfsetispeed( &new_setup, baud_code );

		/* Set these new parameters. */
		tcsetattr( port_ref, TCSANOW,&new_setup);

		result = 1;
	}

	return result;

}


/********************************************************************/


/* station#1-4 active flag */
static	int	stn_flg[5] = {OFF_PS, OFF_PS, OFF_PS, OFF_PS, OFF_PS};


/* ChangeUnit_ps() changes unit inchi/meter
 *    Returns False (zero) if buffer is full
 *    Returns True (non-zero) if successful
 *    give unit:  inchi: INCHI_PS/ meter: METER_PS.
 */
#define INCHI_PS 0
#define METER_PS 1
int ChangeUnit_ps(int unit);

/* ChangeFormat_ps() changes format ascii/binary
 *    Returns False (zero) if buffer is full
 *    Returns True (non-zero) if successful
 *    give form:  ascii: ASCII_PS/ binary: BINARY_PS.
 */
#define ASCII_PS 0
#define BINARY_PS 1
int ChangeFormat_ps(int form);


int Init_ps(char *port, long int baud)
{
	int result;

	result = OpenSP_com(port, baud);

	if(result == 1){
	        tcflush(port_ref, TCIOFLUSH);

		Reset_ps();
		ChangeFormat_ps(BINARY_PS);
		ChangeUnit_ps(METER_PS);
		SetStation_ps(1, OFF_PS);
		SetStation_ps(2, OFF_PS);
		SetStation_ps(3, OFF_PS);
		SetStation_ps(4, OFF_PS);
	}

	return result;
}


int Close_ps(void)
{
	if(port_ref != -1){
		tcsetattr( port_ref, TCSANOW,&old_setup);
		close(port_ref);
		port_ref = -1;
		return 1;
	} else {
		fprintf(stderr, "ps has not been opened\n");
		return 0;
	}
}


int Reset_ps(void)
{
	char c = CTL_COM('Y');
	int result, i;

	result = (write(port_ref, &c, 1) == 1);
	fprintf(stderr, "Reset_ps: now resetting ps, please wait 15 sec");
	fprintf(stderr, "          \n(NEVER push stylus button now!)");
	/* sleep(15); */
	for(i = 0; i < 15; i ++){
		sleep(1);
		fprintf(stderr, ".");
	}
	fprintf(stderr, "done.\n");

	return result;
}


int ChangeUnit_ps(int unit)
{
	switch(unit){
	    case INCHI_PS:
		return (write(port_ref, "U", 1) == 1);
	    case METER_PS:
		return (write(port_ref, "u", 1) == 1);
	    default:
		fprintf(stderr, "ChangeUnit_ps: please give INCHI_PS/METER_PS.\n");
		return -1;
	}
}


int ChangeFormat_ps(int form)
{
	switch(form){
	    case ASCII_PS:
		fprintf(stderr, "ChangeFormat_ps: ASCII mode is not supported.\n");
		return (write(port_ref, "F", 1) == 1);
	    case BINARY_PS:
		return (write(port_ref, "f", 1) == 1);
	    default:
		fprintf(stderr, "ChangeFormat_ps: please give ASCII_PS/BINARY_PS.\n");
		return -1;
	}
}


int SetStation_ps(int stn, int act)
{
	int result;
	char actstr[] = "l0,0\r";
	char stlstr[] = "e0,0\r";
	char frmstr[] = "O0,2,4,16\r";

	stn_flg[stn] += act;
	actstr[1] += stn;
	if (act != OFF_PS) actstr[3] = '1';

	result = (write(port_ref, actstr, 5) == 5); /* set activation */

	if(act == OFF_PS) return result;

	frmstr[1] += stn;
	if(act == STL_PS){
		stlstr[1] += stn;
		result = (write(port_ref, stlstr, 5) == 5);
		return (write(port_ref, frmstr, 10) == 10); /* set stylus data */
	} else {
		frmstr[6] = '\r';
		return (write(port_ref, frmstr, 7) == 7); /* set normal data */
	}
}


/* GetPos_ps() gets position, orient and stylus button
 *    Returns stylus button is ON/OFF
 *    give p:     position/orient struct.
 */
static int GetPos_ps(int station, PosOri_ps p)
{
	int i, j;
	char stn, err, c, *f;

	f = (char *) p;

	read(port_ref, &c, 1);
	read(port_ref, &stn, 1); /* station number */
	read(port_ref, &err, 1); /* system error code */
	stn = stn - '0';
#if _DEBUG_
	if(c != '0' || stn != station){
		fprintf(stderr, "GetPos_ps(%d)-1: %d.\n", station, c);
		fprintf(stderr, "GetPos_ps(%d)-ST: %d.\n", station, stn);
		fprintf(stderr, "GetPos_ps(%d)-EC: %c.\n", station, err);
	}
#endif

	for(i = 0; i < 6; i++){
		for(j = 0; j < 4; j++){
/*			read(port_ref, f + (4 * i  + 3 - j), 1); for Z8000 / M680XX (IRIS) */
			read(port_ref, f + (4 * i  + j), 1); /* for x86 */
		}
	}

	if (stn_flg[station] == STL_PS){
		read(port_ref, &c, 1);
#if _DEBUG_
		if(c != ' '){
			fprintf(stderr, "GetPos_ps(%d)-STL: %d.\n", station, c);
		}
#endif
		read(port_ref, &c, 1);
		if (c == '1'){
			return (1 << (station - 1));
		}
		return 0;
	}
	return 0;
}


/* Returns 1+2+4+8 if stl#1 + stl#2 + stl#3 + stl#4 are ON (*) */
int PosRequest_ps(PosOri_ps p1, PosOri_ps p2, PosOri_ps p3, PosOri_ps p4)
{
	int btn = 0;

	write(port_ref, "P", 1);

	if(stn_flg[1] != OFF_PS){
		btn  = GetPos_ps(1, p1);
	}
	if(stn_flg[2] != OFF_PS){
		btn += GetPos_ps(2, p2);
	}
	if(stn_flg[3] != OFF_PS){
		btn += GetPos_ps(3, p3);
	}
	if(stn_flg[4] != OFF_PS){
		btn += GetPos_ps(4, p4);
	}

	tcflush(port_ref, TCIOFLUSH);

	return btn;
}


/* T_T T_T T_T T_T T_T T_T T_T T_T T_T T_T */


/* SetSynchMode_ps() changes synchronization mode default//CRT
 *    Returns False (zero) if buffer is full
 *    Returns True (non-zero) if successful
 *    give mode:  default: 0// CRT: 4 (using telephone picup).
 */
int SetSynchMode_ps(int mode);


int SetSynchMode_ps(int mode)
{
	char str[] = "y0\r";

	str[1] += mode;
	return (write(port_ref, str, 3) == 3);
}


/* SetTipOffser_ps() changes offset of stylus
 *    Returns False (zero) if buffer is full
 *    Returns True (non-zero) if successful
 *    give stn:    station number.
 *         x/y/z:  x/y/z-direction tip offset (A.BCD inchi).
 *                 (default: about 2.523, 0.004, 0.030)
 */
int SetTipOffset_ps(int stn, double x, double y, double z);


int SetTipOffset_ps(int stn, double x, double y, double z)
{
	int result;
	char str[256] = "N0\r";

	ChangeUnit_ps(INCHI_PS);
	sprintf(str, "N%d,%5.3f,%5.3f,%5.3f\r", stn, x, y, z);
	result = (write(port_ref, str, 21) == 21);
	ChangeUnit_ps(METER_PS);
	
	return result;
}

