/*********************************************************************

          cg.c: CyberGlove device driver

                                        Kenji with Tail

*********************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <termio.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/time.h>
#include <stropts.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <errno.h>
#include <termios.h>

#include "cg.h"


#define _DEBUG_ 1


/*********          Serial Port comunication driver          ********/
/* Public constants for x86(?)... */
#define OFLAGS  (OPOST | OLCUC | ONLCR | OCRNL | ONOCR  \
                        | ONLRET | OFILL | OFDEL)
#define LFLAGS  (ICANON | ISIG | XCASE | ECHO | ECHOE | ECHOK  \
                        | ECHONL | NOFLSH | TOSTOP | ECHOCTL  \
                        | ECHOPRT | ECHOKE | IEXTEN)
#define IFLAGS  (IGNBRK | BRKINT | IGNPAR | PARMRK | INPCK  \
                        | INLCR | ICRNL | IGNCR | IUCLC | ISTRIP  \
                        | IXON | IXOFF | IMAXBEL)



#define MIN_TIMEOUT 1
#define TIM_TIMEOUT 0

#define CTL_COM(n) ((n) - 0x40)
#define CR_COM 13
#define LF_COM 10
#define NUL_COM 0


#define NUM_COM	(4+1)	/* number of port is 4, 0-4 */

static	int		port_ref[NUM_COM] = {-1, -1, -1, -1, -1};
static	struct	termios	old_setup[NUM_COM];


/* OpenSP_com() opens the given serial port with specified baud rate
 *    Always uses 8 data bits, 1 stop bit, no parity.
 *    Assumes port number is valid and baud rate has been fixed up.
 *    Returns False (zero) if can't open the port.
 */
static int OpenSP_com(int port_id, char *port_name, long int baud)
{
	int		result = 0;	/* Assume the port will not be accessible */
	tcflag_t	baud_code;
	struct	termios	new_setup;

	if (port_ref[port_id] != -1){
		fprintf(stderr, "serial port(%d) has been opened", port_id);
		return result;
	}

	port_ref[port_id] = open(port_name, O_RDWR);

	/* If port and its parameters are accessible, reconfigure for Probe */
	if (port_ref[port_id] >= 0){

		/* Find appropriate baud rate flags */
		switch(baud){
		    case 0L:
			baud_code = B0;		break; /* down ER signal, close */
		    case 50L:
			baud_code = B50;	break;
		    case 75L:
			baud_code = B75;	break;
		    case 110L:
			baud_code = B110;	break;
		    case 134L:
			baud_code = B134;	break;
		    case 150L:
			baud_code = B150;	break;
		    case 200L:
			baud_code = B200;	break;
		    case 300L:
			baud_code = B300;	break;
		    case 600L:
			baud_code = B600;	break;
		    case 1200L:
			baud_code = B1200;	break;
		    case 1800L:
			baud_code = B1800;	break;
		    case 2400L:
			baud_code = B2400;	break;
		    case 4800L:
			baud_code = B4800;	break;
		    case 9600L:
			baud_code = B9600;	break;
		    case 19200L:
			baud_code = B19200;	break;
		    case 38400L:
			baud_code = B38400;	break;
		    case 'A':
			baud_code = EXTA;	break;
		    case 'B':
			baud_code = EXTB;	break;
		    default:
			fprintf(stderr, "OpenSP_com: bad baud rate: %ld\n", baud);
		}

		tcgetattr( port_ref[port_id], &(old_setup[port_id]));
		/* Save original set-up */
		new_setup = old_setup[port_id];

		/* Turn off all post-processing and char translation */
		new_setup.c_oflag = ~OFLAGS;
		new_setup.c_lflag = ~LFLAGS;
		new_setup.c_iflag = ~IFLAGS;

		/* Set for no waiting during char read */
		new_setup.c_cc[VMIN] = MIN_TIMEOUT;
		new_setup.c_cc[VTIME] = TIM_TIMEOUT;

		/* Set for baud rate, 8 data bits, 1 stop bit, no parity */
		new_setup.c_cflag |= CS8 | CREAD;
		cfsetospeed( &new_setup, baud_code );
		cfsetispeed( &new_setup, baud_code );

		/* Set these new parameters. */
		tcsetattr( port_ref[port_id], TCSANOW,&new_setup);

		result = 1;
	}

	return result;

}


/********************************************************************/


/* cg#1-4 get-light flag */
static	int	lit_flg[NUM_COM] = {(int)NULL, OFF_CG, OFF_CG, OFF_CG, OFF_CG};


int Init_cg(int cg_id, char *port, long int baud)
{
	int result;

	result = OpenSP_com(cg_id, port, baud);

	if(result == 1){
	        tcflush(port_ref[cg_id], TCIOFLUSH);

		ReInit_cg(cg_id);
		ReStart_cg(cg_id);
	}

	return result;
}


int Close_cg(int cg_id)
{
	if(port_ref[cg_id] != -1){
		tcsetattr( port_ref[cg_id], TCSANOW,&(old_setup[cg_id]));
		close(port_ref[cg_id]);
		port_ref[cg_id] = -1;
		return 1;
	} else {
		fprintf(stderr, "cg(%d) has not been opened\n", cg_id);
		return 0;
	}
}


int ReInit_cg(int cg_id)
{
	char c = CTL_COM('I');
	int result, i;

	result = (write(port_ref[cg_id], &c, 1) == 1);
	fprintf(stderr, "ReInit_cg: now re-initializing cg(%d), please wait 2 sec", cg_id);
	/* sleep(2); */
	for(i = 0; i < 2; i ++){
		sleep(1);
		fprintf(stderr, ".");
	}

	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != CTL_COM('I')){
		fprintf(stderr, "ReInit_cg(%d)-1: %d.\n", cg_id, c);
	}
#endif
	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != CR_COM){
		fprintf(stderr, "ReInit_cg(%d)-2: %d.\n", cg_id, c);
	}
#endif
	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != LF_COM){
		fprintf(stderr, "ReInit_cg(%d)-3: %d.\n", cg_id, c);
	}
#endif
	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != NUL_COM){
		fprintf(stderr, "ReInit_cg(%d)-4: %d.\n", cg_id, c);
	}
#endif

	fprintf(stderr, "done.\n");

	return result;
}


int ReStart_cg(int cg_id)
{
	char c = CTL_COM('R');
	int result, i;

	result = (write(port_ref[cg_id], &c, 1) == 1);
	fprintf(stderr, "ReStart_cg: now re-starting cg(%d), please wait 3 sec", cg_id);
	/* sleep(3); */
	for(i = 0; i < 3; i ++){
		sleep(1);
		fprintf(stderr, ".");
	}

	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != CTL_COM('R')){
		fprintf(stderr, "ReInit_cg(%d)-1: %d.\n", cg_id, c);
	}
#endif
	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != CR_COM){
		fprintf(stderr, "ReInit_cg(%d)-2: %d.\n", cg_id, c);
	}
#endif
	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != LF_COM){
		fprintf(stderr, "ReInit_cg(%d)-3: %d.\n", cg_id, c);
	}
#endif
	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != NUL_COM){
		fprintf(stderr, "ReInit_cg(%d)-4: %d.\n", cg_id, c);
	}
#endif

	fprintf(stderr, "done.\n");

	return result;
}


int Request_cg(int cg_id, Angle_cg angle)
{
	int i, result;
	char c;

	result = (write(port_ref[cg_id], "G", 1) == 1);

	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != 'G'){			/* above G */
		fprintf(stderr, "Request_cg(%d)-1: %d.\n", cg_id, c);
	}
#endif

	for(i = 0; i < 18; i++){
		read(port_ref[cg_id], (char *) angle + i, 1);
	}
	/* read(port_cg[cg_id], (char *) angle, 18);*/

	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != NUL_COM){
		fprintf(stderr, "Request_cg(%d)-2: %d.\n", cg_id, c);
	}
#endif

	/* get light */
	if(lit_flg[cg_id] == LIT_CG){
		result = (write(port_ref[cg_id], "?L", 2) == 2);
		read(port_ref[cg_id], &c, 1);
#if _DEBUG_
		if(c != '?'){
			fprintf(stderr, "Light_cg(%d)-r1: %d.\n", cg_id, c);
		}
#endif
		read(port_ref[cg_id], &c, 1);
#if _DEBUG_
		if(c != 'L'){
			fprintf(stderr, "Light_cg(%d)-r2: %d.\n", cg_id, c);
		}
#endif
		read(port_ref[cg_id], &c, 1);
		result = (int) c;
		read(port_ref[cg_id], &c, 1);
#if _DEBUG_
		if(c != NUL_COM){
			fprintf(stderr, "Light_cg(%d)-r3: %d.\n", cg_id, c);
		}
#endif
	} else result = lit_flg[cg_id];

	tcflush(port_ref[cg_id], TCIOFLUSH);

	return result;
}


int Light_cg(int cg_id, int req)
{
	int result;
	char c, s[] = "L0";

	lit_flg[cg_id] = req;

	switch(req){
	    case OFF_CG:
	    case ON_CG:
		s[1] = (char) req;
		result = (write(port_ref[cg_id], s, 2) == 2);
		read(port_ref[cg_id], &c, 1);
#if _DEBUG_
		if(c != 'L'){
			fprintf(stderr, "Light_cg(%d)-1: %d.\n", cg_id, c);
		}
#endif
		read(port_ref[cg_id], &c, 1);
#if _DEBUG_
		if(c != NUL_COM){
			fprintf(stderr, "Light_cg(%d)-2: %d.\n", cg_id, c);
		}
#endif
		break;
	    case LIT_CG:
		s[0] = 'J';
		s[1] = 1;
		result = (write(port_ref[cg_id], s, 2) == 2);
		read(port_ref[cg_id], &c, 1);
#if _DEBUG_
		if(c != 'J'){
			fprintf(stderr, "Light_cg(%d)-3: %d.\n", cg_id, c);
		}
#endif
		read(port_ref[cg_id], &c, 1);
#if _DEBUG_
		if(c != NUL_COM){
			fprintf(stderr, "Light_cg(%d)-4: %d.\n", cg_id, c);
		}
#endif
		break;
	    default:
		fprintf(stderr, "Light_cg: please give OFF_CG/ON_CG/LIT_CG.\n");
	}

	return result;
}


int Vibrate_cg(int cg_id, int fin, int vib)
{
	int result;
	char c, s[] = "A100";

	s[1] = (char) 1;
	s[2] = (char) fin;
	s[3] = (char) vib;

	result = (write(port_ref[cg_id], s, 4) == 4);
	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != 'A'){
		fprintf(stderr, "Vibrate_cg(%d)-1: %d.\n", cg_id, c);
	}
#endif
	read(port_ref[cg_id], &c, 1);
#if _DEBUG_
	if(c != NUL_COM){
		fprintf(stderr, "Vibrate_cg(%d)-2: %d.\n", cg_id, c);
	}
#endif
	return result;
}
  

